/*
 *  This file is part of todolist, which is Copyright (C) 1999 Jean-Jacques Moreau
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software Foundation,
 *  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

/* --------------------------------------------------------------------
   Project: Checklist Manager
   Module:  TODOLIST.C
   Author:  Jean-Jacques Moreau
   Started: 11. Aug. 99
   Subject: Program that let's HP100/200LX databases be used as checklists
   -------------------------------------------------------------------- */

/* --------------------------------------------------------------------
                          standard includes
   -------------------------------------------------------------------- */
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>


/* --------------------------------------------------------------------
                           local includes
   -------------------------------------------------------------------- */
#include "pal.h"
#include "palsmsvc.h"


/* --------------------------------------------------------------------
                           global constants
   -------------------------------------------------------------------- */
#define KEYDEPTH   9


/* --------------------------------------------------------------------
                           global variables
   -------------------------------------------------------------------- */


/* --------------------------------------------------------------------
                                menus
   -------------------------------------------------------------------- */
typedef enum {
   CMD_ABOUT = 1,
   CMD_OPEN, CMD_EXIT,
   CMD_TOGGLEITEM, CMD_CLEARALL, CMD_SETALL,
   CMD_SWITCHVIEW, CMD_SUBSET, CMD_SHOWDETAIL, CMD_ZOOM
} MENUCOMMANDS;


/* the 'File' pulldown: Items and Menu */
MENUITEM FileItems[] = {
   { "&Open...",      MA_VAL, MENUVAL(CMD_OPEN) },
   { "E&xit",         MA_VAL, MENUVAL(CMD_EXIT) },
};
MENU FileMenu = { NULL, MS_PULLDN, 0, 0, 2, 2, FileItems };


/* the 'Tick' pulldown: Items and Menu */
MENUITEM TickItems[] = {
   { "&Toggle",        MA_VAL, MENUVAL(CMD_TOGGLEITEM) },
   { "&Clear All",     MA_VAL, MENUVAL(CMD_CLEARALL)   },
   { "&Set All",       MA_VAL, MENUVAL(CMD_SETALL)     },
};
MENU TickMenu = { NULL, MS_PULLDN, 0, 0, 3, 3, TickItems };

/* the 'View' pulldown: Items and Menu */
MENUITEM ViewItems[] = {
   { "S&witch View",   MA_VAL, MENUVAL(CMD_SWITCHVIEW) },
   { "Show &Detail",   MA_VAL, MENUVAL(CMD_SHOWDETAIL) },
   { "&Subset...",     MA_VAL, MENUVAL(CMD_SUBSET)     },
   { "&Zoom",          MA_VAL, MENUVAL(CMD_ZOOM)       },
};
MENU ViewMenu = { NULL, MS_PULLDN, 0, 0, 4,4, ViewItems };

/* the 'Help' pulldown: Items and Menu */
MENUITEM AboutItems[] = {
   { "&About...",     MA_VAL, MENUVAL(CMD_ABOUT)},
};
MENU AboutMenu = { NULL, MS_PULLDN, 0, 0, 1, 1, AboutItems };

/* the menu bar  */
MENUITEM BarItems[] = {
   { "&File",         MA_MNU, &FileMenu         },
   { "&Ticks",        MA_MNU, &TickMenu         },
   { "&View",         MA_MNU, &ViewMenu         },
   { "&Quit",         MA_VAL, MENUVAL(CMD_EXIT) },
   { "&Help",         MA_MNU, &AboutMenu        },
};
MENU BarMenu = { NULL, MS_HORIZ|MS_TOPLVL, 0,0,5,5, BarItems };


/* --------------------------------------------------------------------
                              dialogs
   -------------------------------------------------------------------- */
#define CMSUBSETS 0
#define BNCANCEL 1
#define BNOK 2

char CBSubsetsList[1024];
IHCBINIT CBSubsets = {
   {"",  32, MEDIUM_FONT },
   &CBSubsetsList
};

DLGITEM SubsetDlgItems[] = {
   /*X   Y   W   D   FLAGS     LABEL       INIT       TYPE           SIZE        PRIV */
   { 69, 16,172, 16,   0L, "Subsets ",  &CBSubsets,  IhCombo , sizeof(IHCOMBO ), NULL },
   {169, 56,120, 25,   0L, "  Cancel  ", NULL,       IhButton, sizeof(IHBUTTON), NULL },
   { 29, 56,120, 25,   0L, "    OK    ", NULL,       IhButton, sizeof(IHBUTTON), NULL }
};

DIALOG SubsetDlg = {
   354, 106,
   0L,
   2,
   3,
   SubsetDlgItems,
   DhStandard,
   NULL,
   NULL,
   0
};


/* --------------------------------------------------------------------
                           Functions
   -------------------------------------------------------------------- */

void About() {
   PALWIN *pWin;
   WORD Key;
   char *KeyLabels[10] = {
      NULL, NULL, NULL, NULL, NULL,
      NULL, NULL, NULL, "Cancel", "OK"
   };

   pWin  = OpenWin(WS_HP200, 80, 35, 585, 145, "About Todo List");
   WinTextAlignH(pWin, 10, "HP200LX ToDo List Organizer 1.0", MEDIUM_FONT, TAS_CENTER);
   WinTextAlignH(pWin, 25, "This program is Freeware.", MEDIUM_FONT, TAS_CENTER);
   WinTextAlignH(pWin, 50, "(c) 1999-2000 Jean-Jacques Moreau.", MEDIUM_FONT, TAS_CENTER);
   WinTextAlignH(pWin, 65, "jean-jacques.moreau@wanadoo.fr", MEDIUM_FONT, TAS_CENTER);
   WinTextAlignH(pWin, 80, "Contact me for praise or bug reports", MEDIUM_FONT, TAS_CENTER);

   ShowFKeys(KeyLabels);
   do {
      Key = GetKey();
      if(Key & 0xff)
         Key &= 0xff;
      if (Key != ESC_KEY && Key != F10_KEY)
         m_beep();
   } while (Key != ESC_KEY && Key != F10_KEY);

   CloseWin(pWin);
}

void InitSubsetList(DBDESC *pDD) {
   char *pViewNames;
   int i;

   pViewNames = CBSubsetsList;
   *pViewNames = 0;
   for(i = 0; i < pDD->NumViews; i++) {
      if (ChangeViewPoint(pDD, i) == TRUE) {
         strcpy(pViewNames, pDD->pVDEF->Name);
         pViewNames += strlen(pDD->pVDEF->Name);

         if (i != pDD->NumViews-1) {
            strcpy(pViewNames, "|");
            pViewNames++;
         }

         *pViewNames = 0;
      }
   }
   ChangeViewPoint(pDD, 0);
}

void ChangeSubset(DBDESC *pDD) {
   int i;

   char *KeyLabels[10] = {
      NULL, NULL, NULL, NULL, NULL,
      NULL, NULL, NULL, "Cancel", "OK"
   };

   InitDialog(&SubsetDlg);
   ShowFKeys(KeyLabels);
   SetDlgItem(&SubsetDlg, CMSUBSETS, EDSI_TXT, pDD->pVDEF->Name);

   ShowDialogCentered(&SubsetDlg, "View Subset");
   HandleDialog(&SubsetDlg, 0);

   GetDlgItem(&SubsetDlg, CMSUBSETS, CBGI_IDX, &i);
   CloseDialog(&SubsetDlg);

   if (i != -1) {
      ChangeViewPoint(pDD, i);
      ShowTab(pDD->pVDEF->Name);
   }
}

void ShowKeys(DBDESC *pDD) {
   char *AllKeyLabels[10] = {
      NULL, "Toggle", "SetAll", "ClrAll", "All",
      "Subset", "Zoom+/-", "Detail", "Open", "Quit"
   };

   char *TickedKeyLabels[10] = {
      "About", "Toggle", "SetAll", "ClrAll", "Ticked",
      "Subset", "Zoom+/-", "Detail", "Open", "Quit"
   };

   char *AllButTickedKeyLabels[10] = {
      "About", "Toggle", "SetAll", "ClrAll", "UnTck'd",
      "Subset", "Zoom+/-", "Detail", "Open", "Quit"
   };

   if (!pDD)
      ShowFKeys(AllKeyLabels);
   else if (pDD->XTickMode == NO_FILTER)
      ShowFKeys(AllKeyLabels);
   else if (pDD->XTickMode == TICKS_FILTER)
      ShowFKeys(TickedKeyLabels);
   else if (pDD->XTickMode == ALLBUT_FILTER)
      ShowFKeys(AllButTickedKeyLabels);
}

void MainLoop() {
   DBDESC *pDD;
   WORD Key;
   int Error;
   int RecNo, MaxRecs, RowNo, MaxRows, TargetRec, TargetRow;

   char *DetailKeyLabels[10] = {
      NULL, NULL, NULL, NULL, NULL,
      "Prev", "Next", "List", NULL, "Quit"
   };

   pDD = NULL;

   ShowTopTime("Checklist:", FALSE);

   do {
      ShowKeys(pDD);

      Key = GetKey();
      if(Key & 0xff)
         Key &= 0xff;

      if(Key == MENU_KEY || Key == AF10_KEY)
         Key = HandleMenu(&BarMenu, 0, 10);

      switch(Key) {
         case CMD_ABOUT: /* about */
            About();
            break;

         case CMD_EXIT: /* exit */
            CloseDatabase(pDD);
            return;

         case F9_KEY:
         case CMD_OPEN: /* open DB */
            CloseDatabase(pDD);

            pDD = OpenDatabase("c.gdb", &Error);
            if (!pDD) {
               MsgBox("Error", " | Could not open database | ", NULL, "  OK  ");
               break;
            }

            RecNo = 0;
            RowNo = 0;
            MaxRecs = NumOfRecords(pDD);
            MaxRows = (MaxRecs >= MaxDisplayableRows(pDD)) ? MaxDisplayableRows(pDD) : MaxRecs;

            ShowDBTitle(pDD, "Checklist");
            ClearRows();
            ShowHeader(pDD);
            ShowRows(pDD, RecNo, RowNo);

            InitSubsetList(pDD);

            break;

         case F6_KEY:
         case CMD_SUBSET: /* switch to subset */
            if (!pDD)
               break;
            ChangeSubset(pDD);
            RecNo = 0;
            RowNo = 0;
            MaxRecs = NumOfRecords(pDD);
            MaxRows = (MaxRecs >= MaxDisplayableRows(pDD)) ? MaxDisplayableRows(pDD) : MaxRecs;
            ClearRows();
            ShowHeader(pDD);
            ShowRows(pDD, RecNo, RowNo);
            break;

         case F2_KEY:
         case SPACE_KEY:
         case CMD_TOGGLEITEM: /* tick/untick */
            if (!pDD)
               break;

            FilterRecAtIndex(pDD, RecNo);
            if (pDD->XTickMode == NO_FILTER) {
               if (pDD->pXTICKS[RecNo] == 1) {
                  ShowTickAtIndex(pDD, RowNo, 1);
               } else {
                  HideTickAtIndex(pDD, RowNo, 1);
               }
            } else {
               MaxRecs = NumOfRecords(pDD);
               MaxRows = (MaxRecs >= MaxDisplayableRows(pDD)) ? MaxDisplayableRows(pDD) : MaxRecs;
               if (RecNo > MaxRecs-1) { /* last record? */
                  RecNo = (RecNo > 0) ? RecNo-1 : 0;
                  RowNo = (RowNo > 0) ? RowNo-1 : MaxDisplayableRows(pDD);
               }
               ClearRows();
               ShowHeader(pDD);
               ShowRows(pDD, RecNo, RowNo);
            }

            break;

         case F5_KEY:
         case CMD_SWITCHVIEW: /* switch filter */
            if (!pDD)
               break;

            if (pDD->XTickMode == NO_FILTER) {
               pDD->XTickMode = TICKS_FILTER;
            } else if (pDD->XTickMode == TICKS_FILTER) {
               pDD->XTickMode = ALLBUT_FILTER;
            } else if (pDD->XTickMode == ALLBUT_FILTER) {
               pDD->XTickMode = NO_FILTER;
            }
            ShowKeys(pDD);
            RecNo = 0;
            RowNo = 0;
            MaxRecs = NumOfRecords(pDD);
            MaxRows = (MaxRecs >= MaxDisplayableRows(pDD)) ? MaxDisplayableRows(pDD) : MaxRecs;
            ClearRows();
            ShowHeader(pDD);
            ShowRows(pDD, RecNo, RowNo);

            break;

         case F3_KEY:
         case CMD_SETALL: /* set all */
            if (!pDD)
               break;

            FilterAllRec(pDD, 1);
            RowNo = 0;
            RecNo = 0;
            MaxRecs = NumOfRecords(pDD);
            MaxRows = (MaxRecs >= MaxDisplayableRows(pDD)) ? MaxDisplayableRows(pDD) : MaxRecs;
            ClearRows();
            ShowHeader(pDD);
            ShowRows(pDD, RecNo, RowNo);

            break;

         case F4_KEY:
         case CMD_CLEARALL: /* clear all */
            if (!pDD)
               break;

            FilterAllRec(pDD, 0);
            RowNo = 0;
            RecNo = 0;
            MaxRecs = NumOfRecords(pDD);
            MaxRows = (MaxRecs >= MaxDisplayableRows(pDD)) ? MaxDisplayableRows(pDD) : MaxRecs;
            ClearRows();
            ShowHeader(pDD);
            ShowRows(pDD, RecNo, RowNo);

            break;

         case F7_KEY:
         case CMD_ZOOM:
         case ZOOM_KEY: /* zoom in or out */
            if (!pDD)
               break;

            pDD->ZoomMode = (pDD->ZoomMode == ZOOM_1) ? ZOOM_3 : ZOOM_1;

            RowNo = 0;
            RecNo = 0;
            MaxRecs = NumOfRecords(pDD);
            MaxRows = (MaxRecs >= MaxDisplayableRows(pDD)) ? MaxDisplayableRows(pDD) : MaxRecs;
            ClearRows();
            ShowHeader(pDD);
            ShowRows(pDD, RecNo, RowNo);

            break;

         case UP_KEY: /* move up */
            if (!pDD)
               break;

            if ((RowNo > 0) && (RecNo == 0)) { /* first record, middle of screen? */
               RowNo--;
               ShowRows(pDD, RecNo, RowNo);
            } else if ((RowNo > 0) && (RecNo > 0)) { /* middle of screen? */
               ShowRowAtIndex(pDD, RecNo, RowNo, 0);
               RecNo--; RowNo--;
               ShowRowAtIndex(pDD, RecNo, RowNo, 1);
               ShowRowTab(pDD, RecNo);
            } else if (RecNo > 0) { /* not at beginning of DB? */
               RecNo--;
               ShowRows(pDD, RecNo, RowNo);
            }

            break;

         case DOWN_KEY: /* move down */
            if (!pDD)
               break;

            if ((RowNo < MaxRows-1) && (RecNo == MaxRecs-1)) { /* last record, middle of screen? */
               RowNo++;
               ShowRows(pDD, RecNo, RowNo);
            } else if ((RowNo < MaxRows-1) && (RecNo < MaxRecs-1)) { /* middle of screen? */
               ShowRowAtIndex(pDD, RecNo, RowNo, 0);
               RecNo++; RowNo++;
               ShowRowAtIndex(pDD, RecNo, RowNo, 1);
               ShowRowTab(pDD, RecNo);
            } else if (RecNo < MaxRecs-1) { /* not at end of DB? */
               RecNo++;
               ShowRows(pDD, RecNo, RowNo);
            }

            break;

         case LEFT_KEY: /* move left */
            if (!pDD)
               break;

            SameRowOnPreviousPage(pDD, RecNo, RowNo, &TargetRec, &TargetRow);
            if (IsRowVisible(pDD, TargetRow)) { /* minimal redisplay */
               ShowRowAtIndex(pDD, RecNo, RowNo, 0);
               ShowRowAtIndex(pDD, TargetRec, TargetRow, 1);
               RowNo = TargetRow;
            } else {
               RowNo = NormalizeRow(pDD, TargetRow);
               ShowRows(pDD, TargetRec, RowNo, 0);
            }
            RecNo = TargetRec;
            ShowRowTab(pDD, RecNo);

            break;

         case RIGHT_KEY: /* move right */
            if (!pDD)
               break;

            SameRowOnNextPage(pDD, RecNo, RowNo, &TargetRec, &TargetRow);
            if (IsRowVisible(pDD, TargetRow)) { /* minimal redisplay */
               ShowRowAtIndex(pDD, RecNo, RowNo, 0);
               ShowRowAtIndex(pDD, TargetRec, TargetRow, 1);
               RowNo = TargetRow;
            } else {
               RowNo = NormalizeRow(pDD, TargetRow);
               ShowRows(pDD, TargetRec, RowNo, 0);
            }
            RecNo = TargetRec;
            ShowRowTab(pDD, RecNo);

            break;

         case PGUP_KEY: /* go one page up */
            if (!pDD)
               break;

            RowOnPreviousScreen(pDD, RecNo, RowNo, &TargetRec, &TargetRow);
            if (IsRowVisible(pDD, TargetRow)) { /* minimal redisplay */
               ShowRowAtIndex(pDD, RecNo, RowNo, 0);
               ShowRowAtIndex(pDD, TargetRec, TargetRow, 1);
               RowNo = TargetRow;
            } else {
               RowNo = NormalizeRow(pDD, TargetRow, 1);
               ShowRows(pDD, TargetRec, RowNo);
            }
            RecNo = TargetRec;
            ShowRowTab(pDD, RecNo);

            break;

         case PGDN_KEY: /* go one page down */
            if (!pDD)
               break;

            RowOnNextScreen(pDD, RecNo, RowNo, &TargetRec, &TargetRow);
            if (IsRowVisible(pDD, TargetRow)) { /* minimal redisplay */
               ShowRowAtIndex(pDD, RecNo, RowNo, 0);
               ShowRowAtIndex(pDD, TargetRec, TargetRow, 1);
               RowNo = TargetRow;
            } else {
               RowNo = NormalizeRow(pDD, TargetRow, 1);
               ShowRows(pDD, TargetRec, RowNo);
            }
            RecNo = TargetRec;
            ShowRowTab(pDD, RecNo);

            break;

         case HOME_KEY: /* go home */
            if (!pDD)
               break;

            if (RecNo > 0) { /* not at beginning already? */
               RecNo = 0;
               RowNo = 0;
               ShowRows(pDD, RecNo, RowNo);
            }

            break;

         case END_KEY: /* go to end */
            if (!pDD)
               break;

            if (RecNo < MaxRecs-1) { /* not at end already? */
               RecNo = MaxRecs-1;
               RowNo = MaxRows-1;
               ShowRows(pDD, RecNo, RowNo);
            }

            break;

         case F8_KEY:
         case ENTER_KEY:
         case CMD_SHOWDETAIL: /* show record details */
            if (!pDD)
               break;

            if (RecNo >= MaxRecs-1)
               break;

            ShowFKeys(DetailKeyLabels);
            ShowRowDetail(pDD, RecNo);

            do {
               Key = GetKey();
               if(Key & 0xff)
                  Key &= 0xff;

               switch(Key) {
                  case F6_KEY:
                  case UP_KEY:
                  case LEFT_KEY: /* previous record */
                     if (RecNo > 0) {
                        RecNo--;
                        ShowRowDetail(pDD, RecNo);
                     }

                     break;

                  case F7_KEY:
                  case DOWN_KEY:
                  case RIGHT_KEY: /* next record */
                     if (RecNo < MaxRecs-1) {
                        RecNo++;
                        ShowRowDetail(pDD, RecNo);
                     }

                     break;

                  case F8_KEY:
                  case ENTER_KEY: /* go back to record list */
                     ClearRows();
                     ShowKeys(pDD);
                     ShowHeader(pDD);
                     RowNo = 0;
                     ShowRows(pDD, RecNo, RowNo);

                     break;

                  case F10_KEY:
                     break;
               }
            } while(Key != ESC_KEY && Key != F10_KEY && Key != F8_KEY && Key != ENTER_KEY);

            break;
      }
   } while(Key != ESC_KEY && Key != F10_KEY);

   CloseDatabase(pDD);
}

void main(void) {
   if(!PalInit(1))
      FatalExit("Init failed - CGAGRAPH not loaded ?", 1);

   MainLoop();

   PalDeInit(1);
}
