#include "vec_math.h"


double dot_p (struct vec_r *v1, struct vec_r *v2)
/*****************************************************************************
 *  returns the dot product of vector v1 and v2                              *
 ****************************************************************************/{

    return (v1->x*v2->x + v1->y*v2->y + v1->z*v2->z);
} /* dot_p */


void cross_p (struct vec_r *v1, struct vec_r *v2, struct vec_r *v3)
/*****************************************************************************
 *  computes the cross (vector) product of v1 and v2 in v3                   *
 ****************************************************************************/{

    double a, b, c, d, e, f;

    a = v1->x;
    b = v1->y;
    c = v1->z;
    d = v2->x;
    e = v2->y;
    f = v2->z;

    v3->x =  b * f - c * e;
    v3->y =  c * d - a * f;
    v3->z =  a * e - b * d;
} /* cross_p */


double angle (struct vec_r *v1, struct vec_r *v2)
/*****************************************************************************
 *  returns the angle between v1 and v2,                                     *
 *  angle returned is between 0 and 90                                       *
 ****************************************************************************/{
    double mod1, mod2, dot_prod;

    mod1 = modular (v1);
    mod2 = modular (v2);

    dot_prod = dot_p (v1, v2);

    if (mod1 * mod2 != 0)
        return (acos (dot_prod / (mod1 * mod2)));
    else
        return (0);
} /* angle */


double modular (struct vec_r *v)
/*****************************************************************************
 *  returns the modular (length) of v                                         *
 ****************************************************************************/{

     return (sqrt (v->x * v->x + v->y * v->y + v->z * v->z));
} /* modular */


void prn_vec_rect (struct vec_r *rect, char *msg, FILE *stream)
/*****************************************************************************
 *  prints vector in rectalgular form                                        *
 *  if stream is not NULL, print to stream                                   *
 ****************************************************************************/{

    if (stream == NULL)
        printf ("rect  (% 15.8f, % 15.8f, % 15.8f) %s\n",
                 rect->x, rect->y, rect->z, msg);
    else
        fprintf (stream, "rect  (% 15.8f, % 15.8f, % 15.8f) %s\n",
                          rect->x, rect->y, rect->z, msg);
} /* prn_vec_rect */


void prn_vec_polar (struct vec_p *polar, char *msg, FILE *stream)
/*****************************************************************************
 *  prints vector in polar form                                              *
 *  if stream is not NULL, print to stream                                   *
 ****************************************************************************/{

    if (stream == NULL)
        printf ("polar (% 15.8f, % 15.8f, % 15.8f) %s\n",
                 polar->alpha / M_PI * 180, polar->theta / M_PI * 180, polar->r, msg);
    else
        fprintf (stream, "polar (% 15.8f, % 15.8f, % 15.8f) %s\n",
                 polar->alpha / M_PI * 180, polar->theta / M_PI * 180, polar->r, msg);
} /* prn_vec_polar */


void rect2polar (struct vec_r *rect, struct vec_p *polar)
/*****************************************************************************
 *  converting rectangular vector to polar vector                            *
 ****************************************************************************/{

    double x, y, z, temp;

    x = rect->x;
    y = rect->y;
    z = rect->z;

    polar->r = sqrt (x*x + y*y + z*z);

    if (x >= 0) {                            /* in first and fourth quadrant */
        if (x != 0) {
            if (y >= 0) {                               /* in first quadrant */
                if (y / x < 1e6) {
                    polar->alpha = atan (y / x);
                } else {
                    polar->alpha = M_PI_2;
                }
            } else {                                   /* in fourth quadrant */
                if (-y / x < 1e6) {
                    polar->alpha = atan (y / x);
                } else {
                    polar->alpha = M_PI_2 + M_PI;
                }
                polar->alpha = atan (y / x) + 2 * M_PI;
            }
        } else {                                                /* on y axis */
            if (y >= 0) {
                polar->alpha = M_PI_2;
            } else {
                polar->alpha = M_PI_2 + M_PI;
            }
        }
    } else {                                 /* in second and third quadrant */
        if (fabs (y / x) < 1e6) {
            polar->alpha = atan (y / x) + M_PI;
        } else {
            if (y >= 0) {
                polar->alpha = M_PI_2;
            } else {
                polar->alpha = M_PI_2 + M_PI;
            }
        }
    }


    temp = sqrt (x*x + y*y);
    if (temp != 0.) {
        if (fabs (z / temp) < 1e6) {
            polar->theta = atan (z / temp);
        } else {
            if (z >= 0) {
                polar->theta =  M_PI_2;
            } else {
                polar->theta = -M_PI_2;
            }
        }
    } else {
        if (z >= 0) {
            polar->theta =  M_PI_2;
        } else {
            polar->theta = -M_PI_2;
        }
    }
} /* rect2polar */


void polar2rect (struct vec_p *polar, struct vec_r *rect)
/*****************************************************************************
 *  converting polar vector to rectangular vector                            *
 ****************************************************************************/{

    double alpha, theta, r;

    r = polar->r;
    alpha = polar->alpha;
    theta = polar->theta;

    rect->x = r * cos (theta) * cos (alpha);
    rect->y = r * cos (theta) * sin (alpha);
    rect->z = r * sin (theta);
} /* polar2rect */
