
/* PCjr Cartridge Dumper
   Michael B. Brutman
   Original: July 26th, 2001

   Unofficial support: mbbrutman@yahoo.com

   2005-12-31: Add option to not write header.
   2003-02-23: Add header to beginning of files.  Official release of 1.0
*/


#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <dos.h>
#include <time.h>


// Prototypes

void dumpSystemROM( void );
void dumpROM( unsigned int startSegment, unsigned long int length,
	      char * filename, char *desc );
void scanOptionalRoms( void );
void cartridgeInfo( unsigned char far * const base, char *name );


// Constants

const unsigned char FIRST_BYTE = 0x55;
const unsigned char SECOND_BYTE = 0xAA;

unsigned int ROMareaStart = 0xC000;
unsigned int CartAreaStart = 0xD000;
unsigned int SystemRomStart = 0xF000;


const unsigned long int ROMincrement = 2048ul;


char headerStr[] = "PCJRCART v1.1 ROM Extension dump";
char headerStr2[] = "PCJRCART v1.1 System ROM dump";
char headerStr3[] = "PCJRCART v1.1 Arbitrary memory dump";


// Globals

int makeFiles = 1; // Write files if a cartridge ROM is found
int noHeaders = 0;
int debug     = 0;

int dumpArea  = 0;
unsigned int StartSeg = 0x0;
unsigned int DumpLen = 0x0;
int ROMsFound = 0;



// This makes working with huge pointers a litte easier.
typedef union {
  struct {
    unsigned int off;
    unsigned int seg;
  } c;
  unsigned char huge * huge_ptr;
  unsigned long int li;
} ptr_t;




void main( int argc, char *argv[] ) {

  int dumpRom = 0; // Will not be dumping system ROM

  puts("\nPCJRCART v1.1 by Michael Brutman (mbbrutman@yahoo.com)\n");


  for (int i=1; i<argc; i++) {
    if ( stricmp(argv[i], "-nofiles") == 0 ) {
      makeFiles = 0;
    }
    else if ( stricmp(argv[i], "-h") == 0 ) {
      puts("This program looks for PCjr cartridges or ROM BIOS extensions.");
      puts("While originally written for the PCjr, it will work on any IBM PC.");
      puts("");
      puts("Format is: pcjrcart <options>");
      puts("");
      puts("  -h           for help");
      puts("  -dumpROM     to dump out the system ROM area (F000:0000 to FFFF:000F)");
      puts("  -dumpArea <start_seg> <len in KB>   to dump arbitrary areas" );
      puts("  -nofiles     scans for ROM extensions but won't write them to files");
      puts("  -noheaders   write files without the 128 byte header (not recommended)");
      puts("  -debug       show verbose debugging messages" );
      puts("");
      puts("  Hint: If your cartridge is not found you may need to dump the system ROM");
      puts("  or dump all of the ROM area using -dumpArea 0xC000 192");
      puts("");
      puts("  Unless -noheaders is used, each file has a 128 byte header added to");
      puts("  the beginning of the file to describe the file.");
      exit(0);
    }
    else if ( stricmp(argv[i], "-dumpROM") == 0 ) {
      dumpRom = 1;
    }
    else if ( stricmp(argv[i], "-debug") == 0 ) {
      debug = 1;
    }
    else if ( stricmp(argv[i], "-noheaders") == 0 ) {
      noHeaders = 1;
    }
    else if ( stricmp(argv[i], "-dumpArea") == 0 ) {

      if ( i+2>=argc ) {
	puts("You need to provide a starting segment and a length.");
	exit(-1);
      }
      i++;
      int rc = sscanf( argv[i], "0x%x", &StartSeg );
      if ( rc != 1 ) {
	puts("Start segment needs to be in hexadecimal with a 0x prefix." );
	exit(-1);
      }
      i++;
      rc = sscanf( argv[i], "%d", &DumpLen );
      if ( rc != 1 ) {
	puts("Length needs to be decimal in KB." );
	exit(-1);
      }

      dumpArea = 1;
    }

    else {
      puts("Unknown option");
      exit(-1);
    }
  }

  if ( (dumpRom == 1) && (dumpArea == 1 ) ) {
    puts("You can't specify -dumpRom and -dumpArea together." );
    exit(-1);
  }


  puts("  For help options use the -h option\n");


  if ( dumpRom == 1 ) {
    dumpROM( 0xF000, 64*1024l, "SYS_ROM.BIN", headerStr2 );
  }
  else if ( dumpArea == 1 ) {

    unsigned long int tmpStart = StartSeg * 16l;
    unsigned long int tmpLen = DumpLen*1024l;

    if ( (tmpStart + tmpLen) > 1024*1024l ) {
      puts("You are trying to dump data past the 1MB boundary.");
      exit(-1);
    }

    char name[12];
    sprintf(name, "%0X.bin", StartSeg);

    dumpROM( StartSeg, tmpLen, name, headerStr3 );
  }
  else {

    scanOptionalRoms( );

    if ( ROMsFound == 0 ) {
      puts("\nNo ROM extensions were found.  Try dumping the system ROM instead.");
    }
    else {
      printf("\nROM extensions found: %d\n", ROMsFound );
    }

  }

}



void dumpROM( unsigned int startSegment, unsigned long int length,
	      char * filename, char *desc ) {

  unsigned char buffer[512];
  unsigned char huge * bufferFarAddr = buffer;

  FILE *fp = fopen( filename, "wb" );

  if ( noHeaders == 0 ) {

    char header[128];
    for ( int i=0; i<128; i++ ) header[i] = 0;


    time_t currentTime = time( NULL );
    struct tm *t = localtime( &currentTime );
    char timeBuffer[30];
    strftime( timeBuffer, 30, "%Y-%m-%d %H:%M:%S", t );

    sprintf( header, "Filename: %s\r\nCreated: %s\r\nDesc: %s\r\nStart: %04X:0000, len 0x%lX\r\n\032",
	     filename, timeBuffer, desc, startSegment, length );

    fwrite( header, 128, 1, fp );
  }


  while ( length  ) {

    void far * p = MK_FP( startSegment, 0 );

    _fmemcpy( bufferFarAddr, p, 512 );
    fwrite( buffer, 512, 1, fp );

    startSegment += 32;
    length -= 512;

  }

  fclose(fp);
}



int isSignaturePresent( unsigned int base ) {

  int rc;

  unsigned char far * p = (unsigned char far *)MK_FP( base, 0x0 );

  if ( debug ) {
    printf( "Seg: %04X: ", base );
    for ( int i=0; i < 16; i++ ) {
      printf( "%02X ", *(p+i) );
    }
  }

  unsigned char firstByte  = *(p);
  unsigned char secondByte = *(p+1);

  if ( (firstByte != FIRST_BYTE) || (secondByte != SECOND_BYTE) ) {
    if ( debug ) {
      printf( " - not a ROM.\n");
    }
    rc = 0;
  }
  else {
    if ( debug ) {
      printf( " - is a ROM.\n" );
    }
    rc = 1;
  }

  return rc;
}


unsigned int parseRom( unsigned int seg ) {

  unsigned char far * base = (unsigned char far *)MK_FP( seg, 0x0 );

  unsigned char blocks = *(base+2);

  unsigned long int length = ((unsigned long)blocks)*512ul;

  printf("  Length: %ld\n", length );

  if ( length > 65536ul ) {
    printf("  Warning: ROM length seems bogus.\n");
  }


  if ( (*(base+3) == 0xCB) &&
       (*(base+4) == 0xAA) &&
       (*(base+5) == 0x55) &&
       (*(base+6) == 0x00) &&
       ( (*(base+7) == 0xFF) || (*(base+7) == 0xFE) ) )
  {
    printf( "  Cartridge contains interpreted BASIC code.\n" );

    if ( *(base+7) == 0xFF ) {
      printf( "    The BASIC code is not protected.\n" );
    }
    else {
      printf( "    The BASIC code is protected.\n" );
    }
  }
  else {

    if ( *(base+6) == 0x00 ) {
      printf( "  ROM is bootable.\n" );
    }
    else {
      printf( "  ROM may contain DOS commands.\n" );

      long int index = 6;

      while ( *(base+index) != 0 ) {

	int nameLen = *(base+index);

	if (nameLen > 8 ) {
	  printf( "  Warning: length of DOS command is %d characters.\n", nameLen );
	  printf( "  This ROM probably does not have DOS commands.\n" );
	  break;
	}
	else {
	  char nameBuffer[9];
	  _fmemcpy(nameBuffer, (base+index+1), nameLen );
	  nameBuffer[nameLen] = 0;

	  // Index is on the cmdLen, so add 1 to get to the name,
	  // add the length of the name, then add 1 to skip past
	  // the first byte of the jmp instruction.

	  long int jumpInstIndex = index + 1 + nameLen;
	  unsigned char jumpInst = *(base+jumpInstIndex);

	  if ( jumpInst == 0xE9 ) {

	    // Ending arithmetic is to compute the proper entry point,
	    // which is relative to the jump instruction.

	    unsigned int jumpLoc = *((unsigned int huge *)(base+jumpInstIndex+1)) + jumpInstIndex + 3;

	    printf( "  Command: %8s  Code: %04x\n", nameBuffer, jumpLoc );
	  }
	  else {
	    // Hmm, that wasn't a jump instruction.  Just give the command.
	    printf( "  Command: %8s  Code: unknown\n", nameBuffer );
	  }
	}

	index = index + nameLen + 4;

      } // end while

    } // endif DOS commands

  } // endif BASIC code

  return blocks;
}





void scanOptionalRoms( void ) {

  // According to the technical reference, optional ROMS may
  // be found from C0000 to F0000.  Optional ROMs are a minimum
  // of 2KB in length.

  unsigned int base = ROMareaStart;

  for ( ; base < SystemRomStart; ) {

    int signatureFound = isSignaturePresent( base );

    if ( signatureFound == 0 ) {
      base = base + 128;
      continue;
    }

    int isCart = ( base >= CartAreaStart );

    if (debug == 0 ) {
      printf( "%s signature found at %04x\n",
	      (isCart ? "Cartridge" : "ROM" ),
	      base );
    }

    ROMsFound++;

    unsigned int blocks = parseRom( base );


    if ( makeFiles != 0 ) {

      char name[20];
      sprintf(name, "%0X.bin", base);

      unsigned long int len = blocks * 512l;

      dumpROM( base, len, name, headerStr );

    }
    else {
      puts("  -nofiles option used, not writing image");
    }

    base = base + (ROMincrement/16);

    //Brian code
    //if ( cartridgeLength > 65536 ) {
    //  base = base + ROMincrement;
    //}
    //else {
    //  base = base + cartridgeLength;
    //}

  }

}
